/**
 * \file: h264_nal_sei_hdr.c
 *
 * Builds a NAL SEI header which contains an additional information of the content of the following picture frame
 *
 * author: Timo Wischer / ADIT / SW1 / twischer@de.adit-jv.com
 *
 * copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

/*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdint.h>
#include <string.h>
#include "h264_nal_sei_hdr.h"


//#define DEBUG

#ifndef __packed
#define __packed  __attribute__((packed))
#endif


/* see D.1.6 User data unregistered SEI message syntax */
typedef struct {
    uint8_t uuid_iso_iec_11578[UUID_SIZE];
    /* size of this value is given by payloadSize - sizeof(uuid_iso_iec_11578) */
    uint8_t payload[0];
} __packed user_data_unregistered_t;


/* see 7.3.2.3.1 Supplemental enhancement information message syntax */
typedef enum {
    USER_DATA_UNREGISTERED = 5
} sei_payload_type_t;

typedef struct {
    /* further byte needed, if payloadType>=0xFF */
    sei_payload_type_t payloadType:8;
    /* further byte needed if payloadSize>=0xFF */
    uint8_t payloadSize;
    union {
        /* see D.1 SEI payload syntax */
        user_data_unregistered_t user_data_unregistered;
    } payload;
} __packed sei_message_t;


/* see 7.3.2.3 Supplemental enhancement information RBSP syntax */
enum {
    RBSP_STOP_BYTE = 0x01
};

typedef struct {
    sei_message_t message;

    /* Bit fields only used for documentation,
       because bit order differs on different platforms.
       uint8_t rbsp_stop_one_bit:1 = 1;
       uint8_t rbsp_alignment_zero_bit:7 = 0;

       Be careful, the position of this value is depending
       on the size of the sei message payload size
     */
    uint8_t rbsp_stop_byte;
} __packed sei_rbsp_t;


/* see 7.4.1 NAL unit semantics */
typedef enum {
    SEI_RBSP = 6
} nal_unit_type_t;


/* see 7.3.1 NAL unit syntax */
typedef struct {
    /* see 7.4.1 NAL unit semantics
       Bit fields only used for documentation,
       because bit order differs on different platforms.
       uint8_t forbidden_zero_bit:1 = 0;
       uint8_t nal_ref_idc:2 = 0;
       nal_unit_type_t nal_unit_type:5 = SEI_RBSP;
     */
    uint8_t nal_unit_type;

    /* contains struct sei_rbsp, but includes a 0x03 after each two 0x00
        rbsp_bytes < 0x000003 <<extend<< 0x0000 < sei_rbsp
    */
    uint8_t rbsp_bytes[0];
} __packed nal_unit_t;


/* see B.1.1 Byte stream NAL unit syntax */
typedef struct {
    uint8_t start_code_prefix[4];
    nal_unit_t data;
} __packed byte_stream_nal_unit_t;


static GstBuffer* buffer = NULL;


/* Inserts 0x03 after each 0x00 0x00.
   Returns the size of the converted message
*/
static size_t
h264_nal_sei_hdr_escape(uint8_t *dst, const uint8_t *src, const size_t src_size)
{
    const uint8_t* const orig_dst = dst;
    const uint8_t* const end = &src[src_size];

    if( src < end ) *dst++ = *src++;
    if( src < end ) *dst++ = *src++;
    while( src < end )
    {
        if( src[0] <= 0x03 && !dst[-2] && !dst[-1] )
            *dst++ = 0x03;
        *dst++ = *src++;
    }

    return dst - orig_dst;
}


int
h264_nal_sei_hdr_build(const uint8_t uuid[UUID_SIZE], const uint8_t* const user_data, const size_t size)
{
    h264_nal_sei_hdr_free();

    /* fail, if payload size can not be encoded in one byte.
       Bigger payloads are supported by the NAL units,
       but this feature is not implemented with this build process.
       Size of 255 is also not supported,
       becasue this value is reserved for extended payload */
    if (sizeof(user_data_unregistered_t) + size >= 0xFF) {
        return -1;
    }

    /* allocate local buffer for building sei message */
    uint8_t rbsp_bytes_unescaped[sizeof(sei_rbsp_t) + size];

    sei_rbsp_t* const rbsp_unescaped = (sei_rbsp_t*)rbsp_bytes_unescaped;
    rbsp_unescaped->message.payloadType = USER_DATA_UNREGISTERED;
    rbsp_unescaped->message.payloadSize = sizeof(user_data_unregistered_t) + size;

    memcpy(rbsp_unescaped->message.payload.user_data_unregistered.uuid_iso_iec_11578, uuid, UUID_SIZE);
    memcpy(rbsp_unescaped->message.payload.user_data_unregistered.payload, user_data, size);

    /* the position of the stop byte is depending on the SEI message payload size */
    uint8_t* const real_stop_byte_pos = &(rbsp_unescaped->rbsp_stop_byte) + size;
    *real_stop_byte_pos = RBSP_STOP_BYTE;


    /* In worst case NAL escape will add one byte for each two bytes.
       So add 1/2 bytes for each byte of the unescaped size */
    const size_t sei_rbsp_max_size = (sizeof(sei_rbsp_t) + size) * 3 / 2;
    /* alocate gst buffer with worst case size and reduce later */
    buffer = gst_buffer_new_and_alloc(sizeof(byte_stream_nal_unit_t) + sei_rbsp_max_size);

    byte_stream_nal_unit_t* const nal_unit = (byte_stream_nal_unit_t*)GST_BUFFER_DATA(buffer);
    /* use start code 0x00, 0x00, 0x00, 0x01 */
    memset(nal_unit->start_code_prefix, 0, sizeof(nal_unit->start_code_prefix));
    nal_unit->start_code_prefix[sizeof(nal_unit->start_code_prefix) - 1] = 0x01;
    nal_unit->data.nal_unit_type = SEI_RBSP;

    const size_t sei_rbsp_exact_size = h264_nal_sei_hdr_escape(nal_unit->data.rbsp_bytes, rbsp_bytes_unescaped, sizeof(rbsp_bytes_unescaped));

    /* shrink gst buffer reguarding to the exact SEI RBSP message size */
    GST_BUFFER_SIZE(buffer) = sizeof(byte_stream_nal_unit_t) + sei_rbsp_exact_size;

#ifdef DEBUG
    uint8_t* data = (uint8_t*)GST_BUFFER_DATA(buffer);
    for (int i=0; i<GST_BUFFER_SIZE(buffer); i++) {
        printf("%02x ", data[i]);
    }
    printf("\n");
#endif

    return 0;
}


GstBuffer* const
h264_nal_sei_hdr_buffer()
{
    return buffer;
}


void
h264_nal_sei_hdr_free()
{
    if (buffer != NULL) {
        gst_buffer_unref(buffer);
        buffer =  NULL;
    }
}
